using System;
using System.Data;
using System.Data.SqlClient;
using System.Data.SqlTypes;
using System.Reflection;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using System.Collections;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ANTISERAWORKLIST = gov.va.med.vbecs.Common.VbecsTables.AntiseraWorklist;
using ANTISERAWORKLISTTYPING = gov.va.med.vbecs.Common.VbecsTables.AntiseraWorklistTyping;
using ANTISERAUNITTEST = gov.va.med.vbecs.Common.VbecsTables.AntiseraUnitTest;
using BLOODUNITTEST = gov.va.med.vbecs.Common.VbecsTables.BloodUnitTest;

namespace gov.va.med.vbecs.DAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>05/30/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// AntiseraWorklist data access layer class.
	///</summary>

	#endregion
	
	public class AntiseraWorklist
	{

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="793"> 
		///		<ExpectedInput>AntiseraWorklistGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of Antisera Worklist entries</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="794"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return AntiseraWorklist table entry, based on given AntiseraWorklistGuid.
		/// Used in BOL.AntiseraWorklist constructor.
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera Worklist Unique Identifier</param>
		/// <returns>Data table containing a single antisera worklist entry</returns>
		public static DataTable GetAntiseraWorklistByGuid(System.Guid antiseraWorklistGuid)
		{
			DataSet ds;
		   
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraWorklistByGuid.antiseraworklistguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntiseraWorklistByGuid.divisioncode, System.Data.SqlDbType.Char),
			};

			prms[0].Value = antiseraWorklistGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraWorklistByGuid.StoredProcName, prms);
        	
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="795"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of Partially Completed Worklists</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="796"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return a table containing antisera worklists in the user's division
		/// that have not yet been completed.
		/// </summary>
		/// <returns>Data table of partially completed antisera worklists</returns>
		public static DataTable GetPartiallyCompletedAntiseraWorklists()
		{
			DataSet ds;
		   
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraWorklistsPartiallyCompleted.divisioncode, System.Data.SqlDbType.VarChar),
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraWorklistsPartiallyCompleted.StoredProcName, prms);
        	
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="797"> 
		///		<ExpectedInput>AntiseraWorklistGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of Worlist information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="798"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return data table with antigen typing specificities included in
		/// antisera worklist identified by supplied Guid.
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera Worklist Unique Identifier</param>
		/// <returns>Data table with AntiseraUnitTest and RackLot information</returns>
		public static DataTable GetAntiseraWorklistDetails(System.Guid antiseraWorklistGuid)
		{
			DataSet ds;
		   
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraWorklistDetails.antiseraworklistguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntiseraWorklistDetails.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = antiseraWorklistGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraWorklistDetails.StoredProcName, prms);
        	
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="799"> 
		///		<ExpectedInput>AntiseraWorklistTypingGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of AntiseraWorklistTyping information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="800"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return data table with antigen typing specificity details
		/// identified by supplied Guid.
		/// </summary>
		/// <param name="antiseraWorklistTypingGuid">Antisera Worklist Typing Unique Identifier</param>
		/// <returns>Data table with AntiseraUnitTest and RackLot information</returns>
		public static DataTable GetAntiseraWorklistTypingDetails(System.Guid antiseraWorklistTypingGuid)
		{
			DataSet ds;
		   
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraWorklistTypingDetails.antiseraworklisttypingguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntiseraWorklistTypingDetails.divisioncode, System.Data.SqlDbType.Char),
			};

			prms[0].Value = antiseraWorklistTypingGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraWorklistTypingDetails.StoredProcName, prms);
        	
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="801"> 
		///		<ExpectedInput>AntiseraWorklistGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of AntiseraWorklist units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="802"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return data table with blood units included in
		/// antisera worklist identified by supplied Guid.
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera Worklist Unique Identifier</param>
		/// <returns>Data table with Blood Unit information</returns>
		public static DataTable GetAntiseraWorklistUnits(System.Guid antiseraWorklistGuid)
		{
			DataSet ds;
		   
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraWorklistUnits.antiseraworklistguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntiseraWorklistUnits.divisioncode, System.Data.SqlDbType.Char),
			};

			prms[0].Value = antiseraWorklistGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraWorklistUnits.StoredProcName, prms);
        	
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="803"> 
		///		<ExpectedInput>BloodUnitGuid, AntiseraWorklistTypingGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of AntigenTyping results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="804"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return antigen typing blood unit test results for blood unit identified
		/// by blood unit Guid.
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit Unique Identifier</param>
		/// <param name="antiseraWorklistTypingGuid">Antisera Worklist Typing Unique Identifier</param>
		/// <returns>Blood Unit Test results based on blood unit GUID and antisera worklist GUID</returns>
		public static DataTable GetTestResultsForAntiseraWorklist(System.Guid bloodUnitGuid, System.Guid antiseraWorklistTypingGuid)
		{
			DataSet ds;

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForAntiseraWorklist.bloodunitguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForAntiseraWorklist.antiseraworklisttypingguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetBloodUnitTestResultsForAntiseraWorklist.divisioncode, System.Data.SqlDbType.Char)
			};
			prms[0].Value = bloodUnitGuid;
			prms[1].Value = antiseraWorklistTypingGuid;
			prms[2].Value = Common.LogonUser.LogonUserDivisionCode;
			
			ds = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitTestResultsForAntiseraWorklist.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="805"> 
		///		<ExpectedInput>Dataset, int</ExpectedInput>
		///		<ExpectedOutput>Success boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="806"> 
		///		<ExpectedInput>Dataset, int</ExpectedInput>
		///		<ExpectedOutput>Fail boolean</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to save a new antisera worklist entry. Also saves associated antisera worklist typing, rack lot,
		/// antisera unit test, rack test and blood unit test information.
		/// </summary>
		/// <param name="dsSave"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>Success indicator</returns>
		public static bool Save(DataSet dsSave, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable[] dtArray = new DataTable[dsSave.Tables.Count];
			string[] spArray = new string[dsSave.Tables.Count];
			int count = 0;

			foreach(DataTable dtSave in dsSave.Tables)
			{
				if (dtSave.TableName == Common.VbecsTables.AntiseraWorklist.TableName)
				{
					dtArray[count] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[count] = STOREDPROC.InsertAntiseraWorklist.StoredProcName;
				}
				else if (dtSave.TableName == Common.VbecsTables.AntiseraWorklistTyping.TableName)
				{
					dtArray[count] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[count] = STOREDPROC.InsertAntiseraWorklistTyping.StoredProcName;
				}
				else if (dtSave.TableName == Common.VbecsTables.RackLot.TableName)
				{
					dtArray[count] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[count] = STOREDPROC.InsertRackLot.StoredProcName;
				}
				else if (dtSave.TableName == Common.VbecsTables.AntiseraUnitTest.TableName)
				{
					dtArray[count] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[count] = STOREDPROC.InsertAntiseraUnitTest.StoredProcName;
				}
				else if (dtSave.TableName == Common.VbecsTables.RackTest.TableName)
				{
					dtArray[count] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[count] = STOREDPROC.InsertRackTest.StoredProcName;
				}
				else if (dtSave.TableName == Common.VbecsTables.BloodUnitTest.TableName)
				{
					dtArray[count] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[count] = STOREDPROC.InsertBloodUnitTest.StoredProcName;
				}
				else if (dtSave.TableName == Common.VbecsTables.ExceptionQc.TableName)
				{ 
					// the exception date will be different if there are multiple rows
					// so, use the most recent date for all rows
					if(dtSave.Rows.Count > 1)
					{
						// the most recent date will be last
						DateTime exceptionDate = Convert.ToDateTime(dtSave.Rows[dtSave.Rows.Count - 1][Common.VbecsTables.ExceptionQc.ExceptionDate]);
						//
						for (int idx = 0; idx < (dtSave.Rows.Count - 1); idx++)
						{
							dtSave.Rows[idx][Common.VbecsTables.ExceptionQc.ExceptionDate] = exceptionDate;
						}
					}
					dtArray[count] = dtSave;
					spArray[count] = STOREDPROC.InsertExceptionQC.StoredProcName;
				}
				else if (dtSave.TableName == "ExceptionUnitTest")
				{
					dtArray[count] = dtSave;
					spArray[count] = STOREDPROC.InsertExceptionPatientTest.StoredProcName;
				}
				else if(dtSave.TableName == Common.VbecsTables.WorkloadEvent.TableName)
				{
					dtArray[count] = Common.Utility.AppendLastUpdateInformation(dtSave, lastUpdateFunctionId);
					spArray[count] = STOREDPROC.InsertWorkloadEvents.StoredProcName;
				}
				count++;
			}
			StoredProcedure sproc = new StoredProcedure();
			int retVal = sproc.TransactionalGetValue(spArray, dtArray);

			return(retVal == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="807"> 
		///		<ExpectedInput>4 datatables, lastupdatefunctionid</ExpectedInput>
		///		<ExpectedOutput>Success boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="808"> 
		///		<ExpectedInput>4 datatables, lastupdatefunctionid</ExpectedInput>
		///		<ExpectedOutput>Failure boolean</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public statc function to update rack lot, antisear unit test, rack test and blood unit test
		/// information associated with a partially completed antisera worklist
		/// </summary>
		/// <param name="rackLotTable">Rack Lot data table</param>
		/// <param name="antiseraUnitTestTable">Antisera Unit Test data table</param>
		/// <param name="rackTestTable">Rack Test data table</param>
		/// <param name="bloodUnitTestTable">Blood Unit Test data table</param>
		/// <param name="lastUpdateFunctionId">UC or callig method</param>
		/// <returns>Success indicator</returns>
		public static bool Save(DataTable rackLotTable, DataTable antiseraUnitTestTable, DataTable rackTestTable, DataTable bloodUnitTestTable, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable[] dtArray = new DataTable[4];
			string[] spArray = new string[4];
			//
			dtArray[0] = Common.Utility.AppendLastUpdateInformation(rackLotTable, lastUpdateFunctionId);
			spArray[0] = STOREDPROC.InsertRackLot.StoredProcName;
			//
			dtArray[1] = Common.Utility.AppendLastUpdateInformation(antiseraUnitTestTable, lastUpdateFunctionId);
			spArray[1] = STOREDPROC.InsertAntiseraUnitTest.StoredProcName;
			//
			dtArray[2] = Common.Utility.AppendLastUpdateInformation(rackTestTable,lastUpdateFunctionId);
			spArray[2] = STOREDPROC.InsertRackTest.StoredProcName;
			//
			dtArray[3] = Common.Utility.AppendLastUpdateInformation(bloodUnitTestTable,lastUpdateFunctionId);
			spArray[3] = STOREDPROC.InsertBloodUnitTest.StoredProcName;
			//
			StoredProcedure sproc = new StoredProcedure();
			return (sproc.TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="811"> 
		///		<ExpectedInput>AntiseraWorklistGuid, StatusCode, rowVersion</ExpectedInput>
		///		<ExpectedOutput>Success boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="812"> 
		///		<ExpectedInput>AntiseraWorklistGuid, StatusCode, rowVersion</ExpectedInput>
		///		<ExpectedOutput>Failure boolean</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to update the task status of an antisera worklist, and return
		/// success indicator. Used to indicate that an antisera worklist has been completed.
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera Worklist Unique Identifier</param>
		/// <param name="taskStatusCode">Task Status Code</param>
		/// <param name="rowVersion">Row Version</param>
		/// <returns>Success Indicator</returns>
		public static bool UpdateTaskStatus(Guid antiseraWorklistGuid, char taskStatusCode, byte[] rowVersion)
		{
			SqlParameter[] parameters = {
											new SqlParameter (STOREDPROC.UpdateAntiseraWorklistTaskStatus.antiseraworklistguid, System.Data.SqlDbType.UniqueIdentifier),
											new SqlParameter (STOREDPROC.UpdateAntiseraWorklistTaskStatus.taskstatuscode, System.Data.SqlDbType.Char),
											new SqlParameter (STOREDPROC.UpdateAntiseraWorklistTaskStatus.lastupdateuser, System.Data.SqlDbType.VarChar),
											new SqlParameter(STOREDPROC.UpdateAntiseraWorklistTaskStatus.rowversion, System.Data.SqlDbType.Binary)
										};
			
			parameters[0].Value = antiseraWorklistGuid;
			parameters[1].Value = taskStatusCode;
			parameters[2].Value = Common.LogonUser.LogonUserName;
			parameters[3].Value = rowVersion;

			int returnValue = new StoredProcedure().TransactionalGetValue(STOREDPROC.UpdateAntiseraWorklistTaskStatus.StoredProcName, parameters);

			return (returnValue == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="813"> 
		///		<ExpectedInput>AntiseraWorklistTypingGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of QC test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="814"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return positive and negative control QC test results
		/// for an antigen typing test identified by supplied Guid.
		/// </summary>
		/// <param name="antiseraWorklistTypingGuid">Antisera Worklist Typing Unique Identifier</param>
		/// <returns>Data table containing Rack Lot and Rack Test information</returns>
		public static DataTable GetAntiseraWorklistQCResults(Guid antiseraWorklistTypingGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraWorklistQCResults.antiseraworklisttypingguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntiseraWorklistQCResults.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = antiseraWorklistTypingGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraWorklistQCResults.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="815"> 
		///		<ExpectedInput>antiseraWorklistTypingGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of QC controls for a reagent</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="816"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Public static function to return the pos and neg QC controls for a reagent in an antisera worklist
		/// </summary>
		/// <param name="antiseraWorklistTypingGuid">Antisera Worklist Typing Unique Identifier</param>
		/// <returns>Data table containing Rack Lot information</returns>
		public static DataTable GetAntiseraWorklistQCControls(System.Guid antiseraWorklistTypingGuid)
		{

			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetAntiseraWorklistQCControls.antiseraworklisttypingguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntiseraWorklistQCControls.divisioncode, System.Data.SqlDbType.Char)
			};

			prms[0].Value = antiseraWorklistTypingGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetAntiseraWorklistQCControls.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/17/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="817"> 
		///		<ExpectedInput>antiseraWorklistTypingGuid, lastupdatefunctionid</ExpectedInput>
		///		<ExpectedOutput>Success boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="818"> 
		///		<ExpectedInput>antiseraWorklistTypingGuid, lastupdatefunctionid</ExpectedInput>
		///		<ExpectedOutput>Failure boolean</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Invalidates test results for antigen typing identified by Guid.
		/// </summary>
		/// <param name="antiseraWorklistTypingGuids">Antisera Worklist Typing Unique Identifiers</param>
		/// <param name="antiseraWorklist">Antisera Worklist (optional)</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="dtExceptions">Exception report entries</param>
		/// <returns>Success indicator</returns>
		public static bool InvalidateAntigenTypingTestResults(DataTable antiseraWorklistTypingGuids, DataTable antiseraWorklist, Common.UpdateFunction lastUpdateFunctionId, DataTable dtExceptions)
		{
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();
			//
			if ( antiseraWorklist != null && antiseraWorklist.Rows.Count > 0 )
			{
				spArray.Add(STOREDPROC.UpdateAntiseraWorklistStatus.StoredProcName);
				dtArray.Add(antiseraWorklist);
			}
			//
			for ( int idx = 0; idx < antiseraWorklistTypingGuids.Rows.Count; idx++ )
			{
				// Get the DataTables we need to perform the invalidation, they are:
				// BloodUnitTest, AntiseraUnitTest, RackLot, and RackTest
				//
				SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetAntigenTypingTestsForInactivation.antiseraworklisttypingguid, SqlDbType.UniqueIdentifier)
				};
				prms[0].Value = antiseraWorklistTypingGuids.Rows[idx][Common.VbecsTables.BloodUnitTest.AntiseraWorklistTypingGuid];
				//
				System.Data.DataSet dsDeactivateTestResults = StoredProcedure.GetData(STOREDPROC.GetAntigenTypingTestsForInactivation.StoredProcName, prms);
				//
				DataTable dtBloodUnitTest = dsDeactivateTestResults.Tables[0];
				DataTable dtAntiseraUnitTest = dsDeactivateTestResults.Tables[1];
				DataTable dtRackLot = dsDeactivateTestResults.Tables[2];
				DataTable dtRackTest = dsDeactivateTestResults.Tables[3];
				//
				if( dtBloodUnitTest != null && dtBloodUnitTest.Rows.Count > 0 )
				{
					spArray.Add(STOREDPROC.InvalidateAntigenTypingBloodUnitTest.StoredProcName);
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtBloodUnitTest, lastUpdateFunctionId));
				}
				//
				if ( dtAntiseraUnitTest != null && dtAntiseraUnitTest.Rows.Count > 0 )
				{
					spArray.Add(STOREDPROC.InvalidateAntigenTypingAntiseraUnitTest.StoredProcName);
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtAntiseraUnitTest, lastUpdateFunctionId));
				}
				//
				if( dtRackLot != null && dtRackLot.Rows.Count > 0 )
				{
					spArray.Add(STOREDPROC.InvalidateAntigenTypingRackLot.StoredProcName);
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtRackLot, lastUpdateFunctionId));
				}
				//
				if( dtRackTest != null && dtRackTest.Rows.Count > 0 )
				{
					spArray.Add(STOREDPROC.InvalidateAntigenTypingRackTest.StoredProcName);
					dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtRackTest, lastUpdateFunctionId));
				}
			}
			//
			spArray.Add(STOREDPROC.InsertExceptionAntigenTest.StoredProcName);
			dtArray.Add(dtExceptions);
			//
			int returnValue = new StoredProcedure().TransactionalGetValue(spArray, dtArray);

			return (returnValue == 0);
		}

	}
}
